import os
import json
from abc import abstractmethod
from typing import Any, Callable, List, Tuple, Dict, Optional

import func_timeout
import tqdm
import trimesh
import torch
import torch.utils.data
import torch_geometric.data
import torch_geometric.transforms
import trimesh.visual.color
import objaverse
import multiprocessing
import glob
import urllib.error

import numpy as np
import pyvista as pv
import point_cloud_utils as pcu
from PIL import Image
from torch.utils.data.dataloader import default_collate
from torch_geometric.data.collate import collate as geom_collate

import utils


def get_all_data_loaders(
    data_config: dict,
    transform: Callable[..., Any] | None = None,
    pre_transform: Callable[..., Any] | None = None,
    pre_filter: Callable[..., Any] | None = None,
    processed_dir_name: str = "processed",
    list_of_files_to_use: List[str] | None = None,
    **kwargs,
) -> Dict:
    if "AmazonBerkeleyObjects" in data_config["root"]:
        dataset_class = AmazonBerkeleyObjectsDataset
        category = None  # The whole dataset is used
    elif "ShapeNet" in data_config["root"]:
        dataset_class = ShapeNetDataset
        category = utils.in_or_default(data_config, "category", "all")
    else:
        avail_datasets = ["AmazonBerkeleyObjects", "ShapeNet"]
        raise NotImplementedError(
            "Which dataset are you using? The available options are:",
            f"{avail_datasets}. Make sure that the name of the",
            "dataset is contained in the 'root' parameter of the config file.",
            "For a new dataset you may attemp to create a new dataset class",
            "inheriting from 'BaseMeshDataset' and overriding the 'download'",
            "method (PS. you can just use a pass statement)!",
        )

    loaders = {}
    for dt in ["train", "val", "test"]:
        dataset = dataset_class(
            root=data_config["root"],
            dataset_type=dt,
            pre_transform=pre_transform,
            transform=transform,
            pre_filter=pre_filter,
            filter_only_files_with=utils.in_or_default(
                data_config, "filter_files_with", []
            ),
            processed_dir_name=processed_dir_name,
            category=category,
            num_workers=utils.in_or_default(data_config, "num_workers", 8),
            list_of_files_to_use=list_of_files_to_use,
        )
        loaders[dt] = MeshLoader(
            dataset,
            data_config["batch_size"],
            shuffle=True,
            pad=utils.in_or_default(data_config, "pad", False),
            num_workers=utils.in_or_default(data_config, "num_workers", 8),
            **kwargs,
        )

    print(f"Files not processed: {len(dataset.all_processing_failure_files)}")
    return loaders


def get_transforms(
    list_transforms: List[str],
    transforms_config: Dict[str, Any] | None,
    root: str | None,
) -> Callable[..., Any] | None:
    if not list_transforms:
        return None
    else:
        cfg = transforms_config if transforms_config is not None else {}
        transforms = []
        for tname in list_transforms:
            if tname == "drop_trimesh":
                transforms.append(DropTrimesh())
            elif tname == "drop_laplacian":
                transforms.append(DropLaplacian())
            elif tname == "drop_edges":
                transforms.append(DropEdges())
            elif tname == "drop_faces":
                transforms.append(DropFaces())
            elif tname == "vertex_colours_from_base_texture":
                transforms.append(
                    VertexColoursFromBaseTexture(root, drop_trimesh=False)
                )
            elif tname == "laplacian_eigendecomposition":
                transforms.append(
                    LaplacianEigendecomposition(
                        mix_lapl_w=utils.in_or_default(cfg, "mix_lapl_w", 0.05),
                        k_eig=utils.in_or_default(cfg, "eigen_number", 10),
                        eps=float(utils.in_or_default(cfg, "eigen_eps", 1e-8)),
                        as_cloud=utils.in_or_default(
                            cfg, "lapl_as_cloud", False
                        ),
                        drop_trimesh=False,
                        store_lapl=utils.in_or_default(
                            cfg, "store_lapl", False
                        ),
                        store_massvec=utils.in_or_default(
                            cfg, "store_massvec", True
                        ),
                        timeout_seconds=utils.in_or_default(
                            cfg, "eigen_timeout_seconds", 300
                        ),
                    )
                )
            elif tname == "scale_invariant_hks":
                transforms.append(
                    ScaleInvariantHeatKernelSignatures(
                        signatures_number=cfg["hks_number"],
                        max_time=utils.in_or_default(cfg, "hks_max_t", 25.0),
                        increment=utils.in_or_default(cfg, "hks_inc_t", 1 / 16),
                        time_scaler=utils.in_or_default(
                            cfg, "hks_scale_t", 0.01
                        ),
                    )
                )
            elif tname == "tangent_gradients":
                transforms.append(
                    TangentGradients(
                        as_cloud=utils.in_or_default(
                            cfg, "grads_as_cloud", False
                        ),
                        save_edges=utils.in_or_default(
                            cfg, "save_edges", False
                        ),
                        save_normals=utils.in_or_default(
                            cfg, "save_normals", False
                        ),
                    )
                )
            elif tname == "tangent_gradients_to_sparse_np":
                transforms.append(TangentGradientsToSparseNp())
            elif tname in ["normalise_scale", "normalize_scale"]:
                transforms.append(torch_geometric.transforms.NormalizeScale())
            elif tname == "normals":
                transforms.append(
                    torch_geometric.transforms.GenerateMeshNormals()
                )
            elif tname == "edges":
                transforms.append(
                    torch_geometric.transforms.FaceToEdge(remove_faces=False)
                )
            elif tname == "sample_everything_poisson":
                transforms.append(
                    SampleEverything(
                        root,
                        utils.in_or_default(cfg, "n_poisson_samples", 20_000),
                        utils.in_or_default(cfg, "resize_texture", False),
                    )
                )
            elif tname == "sample_farthest":
                transforms.append(
                    FarthestPointSampling(
                        utils.in_or_default(cfg, "n_farthest_samples", 200)
                    )
                )
            else:
                raise NotImplementedError(f"{tname} not implemented yet.")

        if "drop_trimesh" in cfg:
            transforms.append(DropTrimesh())

        return torch_geometric.transforms.Compose(transforms)


# Load and batch meshes ########################################################
class MeshLoader(torch.utils.data.DataLoader):
    def __init__(
        self,
        dataset,
        batch_size=1,
        shuffle=False,
        pad=False,
        set_affinity_manually: bool = True,
        **kwargs,
    ):
        collater = MeshCollater(pad, batch_size)
        worker_init = worker_init_fn if set_affinity_manually else None
        super(MeshLoader, self).__init__(
            dataset,
            batch_size,
            shuffle,
            collate_fn=collater,
            worker_init_fn=worker_init,
            **kwargs,
        )


def worker_init_fn(worker_id):
    # Compensate for unexpected modification to CPU affinity of Dataloader
    # workers in some environments where pytorch or a dependency appears to
    # unexpectedly modify CPU affinity
    os.sched_setaffinity(0, range(os.cpu_count()))


class MeshCollater:
    def __init__(self, pad: bool = False, batch_size: int = 1) -> None:
        self._pad = pad
        self._batch_size = batch_size

    def __call__(self, data_list):
        if self._batch_size > 1 and not self._pad:
            return self.custom_collate(data_list)
        elif self._batch_size > 1 and self._pad:
            return self.pad_collate(data_list)
        else:
            return self.equal_sized_collate(data_list)

    def equal_sized_collate(
        self, data_list: List[torch_geometric.data.Data]
    ) -> torch_geometric.data.Data:
        if not isinstance(data_list[0], torch_geometric.data.Data):
            raise TypeError(
                f"DataLoader found invalid type: {type(data_list[0])}. "
                f"Expected torch_geometric.data.Data instead"
            )

        keys = [set(data.keys()) for data in data_list]
        keys = list(set.union(*keys))
        batched_data = torch_geometric.data.Data()
        for key in keys:
            attribute_list = [data[key] for data in data_list]
            try:
                batched_data[key] = default_collate(attribute_list)
            except (TypeError, RuntimeError):
                # attributes with incompatible sizes or unknow types are batched
                # as a list
                batched_data[key] = attribute_list
        return batched_data

    def custom_collate(
        self, data_list: List[torch_geometric.data.Data]
    ) -> torch_geometric.data.Batch:
        """
        Custom collation that mostly pack collates (batching of torch_geometric)
        but evals are batched on dim 0. Also values to convert from padded to
        packed batches are precomputed.
        """
        x_sizes = []
        # pytorch geometric batches automatically on the diagonal attributes
        # with 'adj' in their name. Temporarily rename them to avoid changing
        # Data class or pytorch geometric collater.
        for data in data_list:
            self._rename_attribute(data, "grad_x", "grad_x_adj")
            self._rename_attribute(data, "grad_y", "grad_y_adj")
            x_sizes.append(data.x.shape[0])

        batch, slice_dict, inc_dict = geom_collate(
            torch_geometric.data.Batch,
            data_list=data_list,
            increment=True,
            add_batch=True,
            exclude_keys=["evals", "farthest_sampling_mask"],
        )

        self._rename_attribute(batch, "grad_x_adj", "grad_x")
        self._rename_attribute(batch, "grad_y_adj", "grad_y")
        slice_dict["grad_x"] = slice_dict.pop("grad_x_adj")
        slice_dict["grad_y"] = slice_dict.pop("grad_y_adj")
        inc_dict["grad_x"] = inc_dict.pop("grad_x_adj")
        inc_dict["grad_y"] = inc_dict.pop("grad_y_adj")

        # Note that get_example won't work converting tensors to coo
        # batch.grad_x = batch.grad_x.to_torch_sparse_coo_tensor()
        # batch.grad_y = batch.grad_y.to_torch_sparse_coo_tensor()

        batch._num_graphs = len(data_list)
        batch._slice_dict = slice_dict
        batch._inc_dict = inc_dict

        # Precompute info to convert from padded to packed and vice-versa
        batch.max_verts = max(x_sizes)
        to_padded_mask = torch.zeros(
            (batch._num_graphs, batch.max_verts), device=batch.x.device
        ).bool()
        for i, y in enumerate(x_sizes):
            to_padded_mask[i, slice(0, y)] = True
        batch.to_padded_mask = to_padded_mask.flatten()

        batch.to_packed_idx = torch.cat(
            [
                torch.arange(v, device=batch.x.device) + i * batch.max_verts
                for (i, v) in enumerate(x_sizes)
            ],
            dim=0,
        ).long()

        # batch stack attributes that are going to be used in heat-diffusion.
        # Note that evecs are fed also as input, conversion is performed online.
        batch.evals = torch.stack([d.evals for d in data_list], dim=0)

        if "farthest_sampling_mask" in data_list[0].keys():
            farthest_masks_list = [d.farthest_sampling_mask for d in data_list]
            batch.farthest_sampling_mask = farthest_masks_list[0].new_full(
                (batch._num_graphs, batch.max_verts), False
            )
            for i, data in enumerate(data_list):
                slices = [i, slice(0, farthest_masks_list[i].shape[0])]
                batch.farthest_sampling_mask[slices] = farthest_masks_list[i]

        return batch

    def pad_collate(
        self, data_list: List[torch_geometric.data.Data]
    ) -> torch_geometric.data.Data:
        if not isinstance(data_list[0], torch_geometric.data.Data):
            raise TypeError(
                f"DataLoader found invalid type: {type(data_list[0])}. "
                f"Expected torch_geometric.data.Data instead"
            )

        keys = [set(data.keys()) for data in data_list]
        keys = list(set.union(*keys))

        batched_data = torch_geometric.data.Data()

        pos_list = [data.pos for data in data_list]

        padded_dims = list(pos_list[0].shape)
        padded_dims[0] = max([y.shape[0] for y in pos_list if len(y) > 0])
        padded = pos_list[0].new_full((len(pos_list), *padded_dims), 0.0)

        padded_dims_evecs = list(data_list[0].evecs.shape)
        padded_dims_evecs[0] = padded_dims[0]

        batched_data.pos = padded.clone()
        batched_data.x = padded.clone()
        batched_data.evecs = data_list[0].evecs.new_full(
            (len(pos_list), *padded_dims_evecs), 0.0
        )
        batched_data.massvec = data_list[0].massvec.new_full(
            (len(pos_list), padded_dims[0]), 0.0
        )

        for i, data in enumerate(data_list):
            slices = [i, slice(0, pos_list[i].shape[0])]
            batched_data.pos[slices] = pos_list[i]
            batched_data.x[slices] = data.x
            batched_data.evecs[slices] = data.evecs
            batched_data.massvec[slices] = data.massvec

        for key in keys:
            if key in ["pos", "x", "evecs", "massvec"]:
                continue

            attribute_list = [data[key] for data in data_list]
            try:
                batched_data[key] = default_collate(attribute_list)
            except (TypeError, RuntimeError):
                # attributes with incompatible sizes or unknow types are batched
                # as a list
                batched_data[key] = attribute_list
        return batched_data

    @staticmethod
    def _rename_attribute(data, old_name, new_name):
        data._store._mapping[new_name] = data._store._mapping.pop(old_name)


# Mesh Datasets ################################################################
class BaseMeshDataset(torch_geometric.data.Dataset):
    def __init__(
        self,
        root: str,
        dataset_type: str = "train",
        transform: Callable[..., Any] | None = None,
        pre_transform: Callable[..., Any] | None = None,
        pre_filter: Callable[..., Any] | None = None,
        processed_dir_name: str = "processed",
        filter_only_files_with: str | List[str] | None = None,
        list_of_files_to_use: List[str] | None = None,
        **kwargs,
    ):
        self._root = root
        if not hasattr(self, "_category"):
            self._category = ""
        self._dataset_type = dataset_type
        self._processed_dir_name = processed_dir_name

        torch_geometric.data.makedirs(self.processed_dir)

        self._split_fpath = os.path.join(self.processed_dir, "data_split.json")
        self._train_names, self._test_names, self._val_names = self._split_data(
            os.path.join(self.processed_dir, "data_split.json")
        )

        if isinstance(filter_only_files_with, str):
            self._only_files_with(filter_only_files_with)
        elif isinstance(filter_only_files_with, list):
            for filter in filter_only_files_with:
                self._only_files_with(filter)

        if list_of_files_to_use is not None:
            self._train_names = list_of_files_to_use
            self._test_names = list_of_files_to_use
            self._val_names = list_of_files_to_use

        self._processed_files = [f[:-4] + ".pt" for f in self.raw_file_names]

        super().__init__(root, transform, pre_transform, pre_filter)

    @abstractmethod
    def download(self):
        r"""Downloads the dataset to the :obj:`self.raw_dir` folder."""
        raise NotImplementedError

    @property
    def raw_file_names(self) -> List[str]:
        if self._dataset_type == "train":
            file_names = self._train_names
        elif self._dataset_type == "test":
            file_names = self._test_names
        elif self._dataset_type == "val":
            file_names = self._val_names
        else:
            raise Exception("train, val and test are supported dataset types")
        return file_names

    @property
    def processed_file_names(self) -> List[str]:
        return self._processed_files

    @property
    def processed_dir(self) -> str:
        return os.path.join(self._root, self._processed_dir_name)

    @property
    def all_processing_failure_files(self) -> List[str]:
        all_raw_fnames = self._find_filenames()
        all_processed = self._train_names + self._test_names + self._val_names
        return [f for f in all_raw_fnames if f not in all_processed]

    @property
    def raw_dir(self) -> str:
        return self._root

    def get(self, idx: int) -> torch_geometric.data.Data:
        filename = self.processed_file_names[idx]
        return torch.load(os.path.join(self.processed_dir, filename))

    def process(self):
        fnames_failed_transforms = []
        for fname in tqdm.tqdm(
            self.raw_file_names, desc=f"Processing {self._dataset_type} data"
        ):
            out_path = self._get_corresponding_out_fname(fname)

            if not os.path.exists(out_path):
                is_processed = self._process_single_file(fname, out_path)
                if not is_processed:
                    fnames_failed_transforms.append(fname)

        self._update_files_lists(fnames_to_remove=fnames_failed_transforms)
        print(
            f"The following files were not transformed and are not goint to ",
            f"be used: {fnames_failed_transforms}",
        )

    def _process_single_file(self, fname: str, out_path: str) -> bool:
        try:
            file_path = os.path.join(self._root, fname)
            mesh = utils.load_mesh(file_path)

            verts = torch.tensor(
                mesh.vertices, dtype=torch.float, requires_grad=False
            ).contiguous()

            faces = torch.tensor(
                mesh.faces.T, dtype=torch.long, requires_grad=False
            ).contiguous()

            data = torch_geometric.data.Data(
                pos=verts, face=faces, original_trimesh=mesh, raw_path=fname
            )

            # Handle pre_transform failure (e.g. when eigendecomposition
            # fails or takes too long)
            if self.pre_transform is not None:
                data = self.pre_transform(data)
        except (
            TypeError,
            ValueError,
            func_timeout.FunctionTimedOut,
            RuntimeError,
        ):
            return False

        fname_split = fname.split(os.sep)
        if len(fname_split) > 1:
            utils.mkdir(os.path.join(self.processed_dir, *fname_split[:-1]))
        torch.save(data, out_path)
        return True

    def _get_corresponding_out_fname(self, fname: str) -> str:
        return os.path.join(self.processed_dir, fname[:-4] + ".pt")

    def _find_filenames(
        self, file_ext: str | Tuple[str] | None = None
    ) -> List[str]:
        if file_ext is None:
            file_ext = (".ply", ".obj", ".glb")
        root_l = len(self._root)
        files = []
        path_to_walk = os.path.join(self._root, self._category)
        for dirpath, _, fnames in os.walk(path_to_walk):
            for f in fnames:
                if f.endswith(file_ext):
                    absolute_path = os.path.join(dirpath, f)
                    f = absolute_path[dirpath.index(self._root) + root_l + 1 :]
                    files.append(f)
        return files

    def _split_data(
        self, data_split_list_path: str
    ) -> Tuple[List[str], List[str], List[str]]:
        try:
            with open(data_split_list_path, "r") as fp:
                data = json.load(fp)
            train_list = data["train"]
            test_list = data["test"]
            val_list = data["val"]
        except FileNotFoundError:
            all_file_names = self._find_filenames()

            train_list, test_list, val_list = [], [], []
            for i, fname in enumerate(all_file_names):
                if i % 100 <= 5:
                    test_list.append(fname)
                elif i % 100 <= 10:
                    val_list.append(fname)
                else:
                    train_list.append(fname)

            data = {"train": train_list, "test": test_list, "val": val_list}
            with open(data_split_list_path, "w") as fp:
                json.dump(data, fp)
        return train_list, test_list, val_list

    def _only_files_with(self, attribute_condition: str = "base_color_tex"):
        not_used_path = os.path.join(self.processed_dir, "not_used.json")
        unused_key = f"{self._dataset_type}_no_{attribute_condition}"
        # All unused files are stored in a list to avoid recomputing. Try to see
        # if it has already been created with the desired filter.
        try:
            with open(not_used_path, "r") as fp:
                not_used_all = json.load(fp)
            unused = not_used_all[unused_key]
            for f in unused:
                try:
                    self.raw_file_names.remove(f)
                except ValueError:  # not in raw_file_names list already
                    pass

        # The desired filtering has not been done yet. Do it and save it.
        except (FileNotFoundError, KeyError):
            unused = []
            s = f"Filter {self._dataset_type} files with {attribute_condition}"

            # TODO: iterate through data only once and save table with all
            # metadata that can be quickly filtered without reading all files

            for f in tqdm.tqdm(list(self.raw_file_names), desc=s):
                raw_path = os.path.join(self._root, f)
                try:
                    if not os.path.isfile(raw_path):
                        raise FileNotFoundError
                    mesh = utils.load_mesh(raw_path)
                    if attribute_condition == "base_color_tex":
                        base_texture = mesh.visual.material.baseColorTexture
                        if base_texture is None:
                            raise AttributeError
                    elif attribute_condition == "a_texture":
                        material = mesh.visual.material
                        if isinstance(
                            material, trimesh.visual.material.SimpleMaterial
                        ):
                            texture = material.image
                        else:
                            texture = material.baseColorTexture
                        if texture is None:
                            raise AttributeError
                    elif attribute_condition[:4] == "less":
                        # ex. less_100k_verts -> convert 100k in a number
                        mv = int(attribute_condition.split("_")[1][:-1]) * 1000
                        if mesh.vertices.shape[0] > mv:
                            raise ValueError
                    else:
                        raise NotImplementedError(
                            "only 'base_color_tex' and 'less_Nk_verts'",
                            "currently implemented...",
                        )
                except (
                    AttributeError,
                    ValueError,
                    # TypeError,  # ShapeNet often had NoneTypes (w/o material)
                    # FileNotFoundError,  # ShapeNet, not all files in split exist
                    # IndexError,  # ShapeNet, faces and vertices not compatible
                    # AssertionError,  # shapenet-glb, glb views != byte length
                ):
                    unused.append(f)
                    self.raw_file_names.remove(f)

            # Save/append to avoid recomputing
            if os.path.exists(not_used_path):
                with open(not_used_path, "r") as fp:
                    not_used_all = json.load(fp)
                not_used_all[unused_key] = unused
            else:
                not_used_all = {unused_key: unused}
            with open(not_used_path, "w") as fp:
                json.dump(not_used_all, fp)

        print(
            f"The {len(unused)} {self._dataset_type} files with no",
            f"{attribute_condition} were filtered out and will not be used.",
            "If you changed data split you may need to delete 'not_used.json'.",
        )

    def _update_files_lists(self, fnames_to_remove: List[str]):
        if len(fnames_to_remove) > 0:
            with open(self._split_fpath, "r") as fp:
                data = json.load(fp)

            for fname in fnames_to_remove:
                if self._dataset_type == "train":
                    self._train_names.remove(fname)
                elif self._dataset_type == "test":
                    self._test_names.remove(fname)
                elif self._dataset_type == "val":
                    self._val_names.remove(fname)
                self.processed_file_names.remove(fname[:-4] + ".pt")
                data[self._dataset_type].remove(fname)

            with open(self._split_fpath, "w") as fp:
                json.dump(data, fp)

    def len(self) -> int:
        return len(self.processed_file_names)


class AmazonBerkeleyObjectsDataset(BaseMeshDataset):
    def download(self):
        tarname = "abo-3dmodels.tar"
        url = "https://amazon-berkeley-objects.s3.amazonaws.com/archives/"
        root = self._root

        flag_print_message = True
        if "AmazonBerkeleyObjects/original" in root:
            root = root.replace("/AmazonBerkeleyObjects/original", "")
            flag_print_message = False

        print(
            "You are now about to download the AmazonBerkeleyObjects dataset",
            "The compressed archive weights 155 GB. Once extracted it will",
            "occupy approximately the same space. Once processed additional",
            "175 GB will be required to store the processed information. ",
            "Ensure you have at least 330 GB available if you wish to remove",
            "the compressed archive or 505 GB if you wish to keep it. ",
        )

        # Internally, 'download_url' already checks if file exists
        tar_path = torch_geometric.data.download_url(url + tarname, root)

        extraction_folder = os.path.join(root, "AmazonBerkeleyObjects")
        if not os.path.isdir(extraction_folder):
            torch_geometric.data.makedirs(extraction_folder)
            torch_geometric.data.extract_tar(tar_path, extraction_folder)
            print(
                f"Dataset downloaded and extracted in {extraction_folder}.",
                f"You can delete {tar_path} if you wish.",
            )
        else:
            print(
                "It looks like the dataset has already been downloaded.",
                f"If {extraction_folder} does not contain the desired data,",
                "change the 'root' parameter in your config file or delete",
                f"the folder: {extraction_folder}",
            )

        if flag_print_message:
            print(
                "Please, from the next execution modify the 'root' parameter",
                "in your config file to:",
                f"'{os.path.join(extraction_folder, 'original')}'.",
            )


class ShapeNetDataset(BaseMeshDataset):
    def __init__(
        self,
        root: str,
        dataset_type: str = "train",
        category: str | None = None,
        transform: Callable[..., Any] | None = None,
        pre_transform: Callable[..., Any] | None = None,
        pre_filter: Callable[..., Any] | None = None,
        processed_dir_name: str = "processed",
        filter_only_files_with: str | None = None,
        list_of_files_to_use: List[str] | None = None,
        **kwargs,
    ):
        self._category = "" if category in ["all", None] else category
        super().__init__(
            root,
            dataset_type,
            transform,
            pre_transform,
            pre_filter,
            processed_dir_name,
            filter_only_files_with,
            list_of_files_to_use,
            **kwargs,
        )

    def download(self):
        # Automatic download would require extra packages and permissions
        # to access the repo. Provide instructions instead
        print(
            "Request permissions to access the following hugging face repo:",
            "https://huggingface.co/datasets/ShapeNet/ShapeNetCore. ",
            "Then, make sure git-lfs is installed (git lfs install) and clone:",
            "git clone git@hf.co:datasets/ShapeNet/ShapeNetCore.",
        )


# Mesh transformations #########################################################
class DropTrimesh(torch_geometric.transforms.BaseTransform):
    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        data.original_trimesh = None
        return data


class DropLaplacian(torch_geometric.transforms.BaseTransform):
    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        data.lapl = None
        return data


class DropEdges(torch_geometric.transforms.BaseTransform):
    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        data.edge_index = None
        return data


class DropFaces(torch_geometric.transforms.BaseTransform):
    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        data.face = None
        return data


class MixinColourSampling:
    def _initialise_mixin(self, root: str):
        self._root = root

    def _load_trimesh_and_base_texture(
        self, data: torch_geometric.data.Data, merge_tex: bool
    ) -> trimesh.Trimesh:
        if (
            "original_trimesh" not in data.keys()
            or data.original_trimesh is None
        ):
            raw_path = os.path.join(self._root, data.raw_path)
            original_trimesh = utils.load_mesh(raw_path, merge_tex=merge_tex)
        else:
            original_trimesh = data.original_trimesh

        try:
            base_texture = original_trimesh.visual.material.baseColorTexture
            # Note: objects with a trimesh SimpleMaterial are discarded because
            # they can hold a single texture image, which is more likely to have
            # shadows baked in (i.e. it is less likely to be an albedo).
            if base_texture is None:
                raise AttributeError

        except AttributeError:
            if any(x in self._root for x in ["ShapeNet", "shapenet", "averse"]):
                # ShapeNet should only have albedos, which may also be
                # contained in a SimpleMaterial. Be more flexible and try to
                # to get the texture of a SimpleMaterial too.
                # Be more flexible also for Objaverse.
                try:
                    base_texture = original_trimesh.visual.material.image
                    if base_texture is None:
                        raise AttributeError
                except AttributeError:
                    print(
                        "Filter the data setting:"
                        "'filter_only_files_with: a_texture' ",
                        "in the dataset constructor or config file.",
                    )
            else:
                print(
                    "Filter the data setting:"
                    "'filter_only_files_with: base_color_tex' ",
                    "in the dataset constructor or config file.",
                )
        return original_trimesh, base_texture

    def _colours_for_training(self, colours):
        # get only RGB and normalise them in [0, 1]
        colours = colours[:, :3] / 255
        # shift colours in [-1, 1]
        colours = (colours - 0.5) * 2
        return colours


class VertexColoursFromBaseTexture(
    MixinColourSampling, torch_geometric.transforms.BaseTransform
):
    def __init__(self, root: str, drop_trimesh: bool = False) -> None:
        self._drop_trimesh = drop_trimesh
        self._initialise_mixin(root)

    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        if not all(c in data.keys() for c in ("original_trimesh", "texture")):
            original_trimesh, tex_img = self._load_trimesh_and_base_texture(
                data, merge_tex=True
            )
        else:
            original_trimesh, tex_img = data.original_trimesh, data.texture

        # Store absolute path to mesh in case need to retrieve vertices and
        # faces for rendering.
        data.raw_abs_path = os.path.join(self._root, data.raw_path)

        uv = original_trimesh.visual.uv % 1.0
        tex = trimesh.visual.texture.TextureVisuals(uv, image=tex_img)

        cols = self._colours_for_training(tex.to_color().vertex_colors)

        assert cols.shape[0] == data.pos.shape[0]

        data.x = torch.tensor(
            cols, dtype=torch.float, requires_grad=False
        ).contiguous()

        if self._drop_trimesh:
            data.original_trimesh = None
        return data


class LaplacianEigendecomposition(torch_geometric.transforms.BaseTransform):
    def __init__(
        self,
        mix_lapl_w: float = 0.05,
        k_eig: int = 10,
        eps: float = 1e-8,
        as_cloud: bool = False,
        drop_trimesh: bool = False,
        store_lapl: bool = False,
        store_massvec: bool = True,
        timeout_seconds: int = 300,
    ) -> None:
        self._mix_lapl_w = mix_lapl_w
        self._k_eig = k_eig
        self._eps = eps
        self._as_cloud = as_cloud
        self._drop_trimesh = drop_trimesh
        self._store_lapl = store_lapl
        self._store_massvec = store_massvec
        self._timeout_seconds = timeout_seconds

    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        evals, evecs, lapl, massvec = func_timeout.func_timeout(
            timeout=self._timeout_seconds,
            func=utils.compute_eig_laplacian,
            kwargs={
                "verts": np.array(data.original_trimesh.vertices),
                "faces": np.array(data.original_trimesh.faces),
                "mix_lapl_w": self._mix_lapl_w,
                "k_eig": self._k_eig,
                "eps": self._eps,
                "as_cloud": self._as_cloud,
            },
        )
        data.evals = torch.tensor(evals, dtype=torch.float, requires_grad=False)
        data.evecs = torch.tensor(evecs, dtype=torch.float, requires_grad=False)

        if self._store_lapl:
            data.lapl = utils.sparse_np_to_torch(lapl)
        if self._store_massvec:
            data.massvec = torch.tensor(
                massvec, dtype=torch.float, requires_grad=False
            )

        if self._drop_trimesh:
            data.original_trimesh = None
        return data


class ScaleInvariantHeatKernelSignatures(
    torch_geometric.transforms.BaseTransform
):
    def __init__(
        self,
        signatures_number: int,
        max_time: float = 25.0,
        increment: float = 1 / 16,
        time_scaler: float = 0.01,
    ) -> None:
        """Compute scale invariant heat kernel signatures as described in
        'Scale-invariant heat kernel signatures for non-rigid shape recognition'
        by Bronstein and Kokkinos (2010). The default parameters are the same as
        those suggested in the paper.
        """
        self._signatures_number = signatures_number
        self._max_time = max_time
        self._increment = increment
        self._time_scaler = time_scaler
        self._tau_step = int(max_time / increment) + 1
        assert self._tau_step > signatures_number

    def __call__(self, data: Any) -> Any:
        taus = torch.linspace(
            0.0,
            self._max_time,
            steps=self._tau_step,
            device=data.evals.device,
            dtype=data.evals.dtype,
        )
        times = self._time_scaler * 2**taus

        hks = utils.compute_hks(data.evals, data.evecs, times)
        hks += 1e-8  # add small bias to prevent log(0)=-inf

        log_hks = torch.log(hks)
        derivative = log_hks.narrow(
            dim=1, start=1, length=log_hks.size(1) - 1
        ) - log_hks.narrow(dim=1, start=0, length=log_hks.size(1) - 1)

        fft = torch.fft.fft(derivative, dim=1)
        data.hks = torch.abs(fft)[:, : self._signatures_number]
        return data


class TangentGradients(torch_geometric.transforms.BaseTransform):
    def __init__(
        self,
        as_cloud: bool = False,
        save_edges: bool = False,
        save_normals: bool = False,
    ) -> None:
        self._face_to_edge_transform = torch_geometric.transforms.FaceToEdge(
            remove_faces=False
        )
        self._vertex_normals_transform = (
            torch_geometric.transforms.GenerateMeshNormals()
        )
        self._as_cloud = as_cloud
        self._save_edges = save_edges
        self._save_normals = save_normals

    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        if data.edge_index is None:
            data = self._face_to_edge_transform(data)
        if "norm" not in data.keys() or data.norm is None:
            data = self._vertex_normals_transform(data)

        data.grad_x, data.grad_y = utils.get_grad_operators(
            data.pos, data.face, data.edge_index, data.norm, self._as_cloud
        )

        if not self._save_edges:
            data.edge_index = None
        if not self._save_normals:
            data.norm = None
        return data


class TangentGradientsToSparseNp(torch_geometric.transforms.BaseTransform):
    # Pytorch doesn't load data with multiple workers if gradients are sparse
    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        data.grad_x = utils.sparse_torch_to_np(data.grad_x.coalesce())
        data.grad_y = utils.sparse_torch_to_np(data.grad_y.coalesce())
        return data


class FarthestPointSampling(torch_geometric.transforms.BaseTransform):
    def __init__(self, n_points: int = 100) -> None:
        self._n_points = n_points

    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        sampling_mask = utils.farthest_point_sampling(data.pos, self._n_points)
        # sampled_points = data.pos[sampling_mask, :]
        data.farthest_sampling_mask = sampling_mask
        return data


class SampleEverything(
    MixinColourSampling, torch_geometric.transforms.BaseTransform
):
    def __init__(
        self,
        root: str,
        n_samples: int = 20_000,
        resize_texture: bool = False,
        save_original_vertices: bool = True,
    ) -> None:
        self._initialise_mixin(root)
        self._n_samples = n_samples
        self._resize_texture = resize_texture
        self._save_original_vertices = save_original_vertices

    def __call__(
        self, data: torch_geometric.data.Data
    ) -> torch_geometric.data.Data:
        if not all(c in data.keys() for c in ("original_trimesh", "texture")):
            original_trimesh, tex_img = self._load_trimesh_and_base_texture(
                data, merge_tex=True
            )
        else:
            original_trimesh, tex_img = data.original_trimesh, data.texture

        # data.v_pos = data.pos  # save vertex pos as they can be transformed
        # verts = data.v_pos.cpu().detach().numpy()
        verts = np.array(original_trimesh.vertices)
        faces = np.array(original_trimesh.faces)
        uv = original_trimesh.visual.uv

        # Store absolute path to mesh in case need to retrieve vertices and
        # faces for rendering.
        data.raw_abs_path = os.path.join(self._root, data.raw_path)

        # Disk Poisson sampling (which returns a list of face_ids
        # and barycentric coordinates)
        num_samples = self._n_samples
        fid, bc = pcu.sample_mesh_poisson_disk(
            verts, faces, num_samples=num_samples
        )
        if fid.shape[0] < num_samples / 10:
            # print(
            #     f"Increasing sample density for {data.raw_path}"
            #     f"as it only had {fid.shape[0]} points"
            # )
            num_samples *= 10
            fid, bc = pcu.sample_mesh_poisson_disk(
                verts, faces, num_samples=num_samples
            )

        # Compute approximate squared radius used in Poisson disk sampling.
        # Note that barycentric coordinates are sampled from original mesh,
        # but we need to estimate radius in preprocessed mesh as the new_pos
        # are also interpolated from preprocessed positions.
        total_area = utils.compute_tot_area(data.pos, data.face)
        # squared_radius = total_area / (0.7 * torch.pi * num_samples)
        squared_radius = (2 * total_area * 0.7**2) / (num_samples * np.sqrt(3))

        # Estimate texture scaling
        if self._resize_texture:
            new_w, new_h = utils.estimate_poisson_scaled_texture_size(
                faces,
                torch.tensor(uv),
                data.pos,
                squared_radius,
                tex_img.size,
            )
            tex_img = tex_img.resize((new_w, new_h))

        if self._save_original_vertices:
            data.verts = data.pos.clone()

        # Get new positions
        new_pos = pcu.interpolate_barycentric_coords(faces, fid, bc, data.pos)
        data.pos = new_pos.to(torch.float).contiguous()

        # Gather scale information
        data.surface_area = total_area.to(torch.float)
        # data.bbox_sides = utils.compute_bounding_box_sides(new_pos)[None, :]

        # Get new colours
        new_uvs = pcu.interpolate_barycentric_coords(faces, fid, bc, uv)
        # UVs can go outside [0, 1] in shapenet, trimesh should now consider it
        new_uvs = new_uvs % 1.0
        new_cols = trimesh.visual.uv_to_color(new_uvs, tex_img)
        # new_cols = trimesh.visual.uv_to_interpolated_color(new_uvs, tex_img)
        new_cols = self._colours_for_training(new_cols)
        data.x = torch.tensor(
            new_cols, dtype=torch.float, requires_grad=False
        ).contiguous()

        # Get new eigenvectors
        new_evecs = pcu.interpolate_barycentric_coords(
            faces, fid, bc, data.evecs
        )
        data.evecs = new_evecs.to(torch.float).contiguous()

        # Massvectors in heat diffusion equation effectively scale the signal on
        # the mesh to diffuse more towards neighbouring vertices when the area
        # of the triangle defined between them is bigger. Since we are using
        # Poisson Disk sampling, points should be approximately at the same
        # distance. Therefore we set the mass to the approximate area of
        # traingles that could be obtained from new samples. We suppose
        # triangles to be equilater with sides equal to the radius used in
        # Poisson sampling.
        data.massvec = torch.tensor(
            squared_radius * torch.sin(torch.pi / torch.tensor(3)) / 2
        )  # * torch.ones(data.x.shape[0], device=data.x.device)
        mean_n_incident_faces = np.mean(np.unique(faces, return_counts=True)[1])
        data.massvec *= mean_n_incident_faces / 3

        # Get new normals, which are used for gradients computation
        new_norm = pcu.interpolate_barycentric_coords(faces, fid, bc, data.norm)
        data.norm = new_norm.to(torch.float).contiguous()

        # Get new gradients
        # TODO: potentially cache and then interpolate tangent frames if faster.
        data.grad_x, data.grad_y = utils.get_grad_operators(
            data.pos, data.face, data.edge_index, data.norm, as_cloud=True
        )
        return data


################################################################################


if __name__ == "__main__":
    import rendering
    import mitsuba as mi
    import drjit

    import cProfile, pstats

    data_config = {
        "root": "/data/AmazonBerkeleyObjects/original",
        "batch_size": 1,
        "num_workers": 0,
        "filter_files_with": ["base_color_tex", "less_60k_verts"],
        "pre_transforms_list": [
            "normalise_scale",
            "laplacian_eigendecomposition",
            "tangent_gradients",
            "drop_trimesh",
        ],
        "transforms_list": [
            "sample_everything_poisson",
            "sample_farthest",
            "drop_laplacian",
            "drop_edges",
            "drop_faces",
        ],
        "transforms_config": {
            "n_poisson_samples": 5_000,
            "n_farthest_samples": 100,
            "resize_texture": True,
            "mix_laplacian_w": 0.05,
            "lapl_as_cloud": False,
            "eigen_number": 128,
            "eigen_eps": 1e-8,
            "eigen_timeout_seconds": 300,
            "store_lapl": False,
            "store_massvec": False,
            "grads_as_cloud": False,
            "save_edges": True,
            "save_normals": True,
        },
    }

    pre_transform = get_transforms(
        data_config["pre_transforms_list"],
        data_config["transforms_config"],
        data_config["root"],
    )
    transform = get_transforms(
        data_config["transforms_list"],
        data_config["transforms_config"],
        data_config["root"],
    )

    # dataset = ShapeNetDataset(
    #     root=data_config["root"],
    #     dataset_type="train",
    #     category=None,
    #     pre_transform=pre_transform,
    #     transform=transform,
    #     pre_filter=None,
    #     filter_only_files_with=data_config["filter_files_with"],
    #     processed_dir_name="processed",
    # )

    profiler = cProfile.Profile()
    profiler.enable()

    loaders = get_all_data_loaders(data_config, transform, pre_transform)
    batch = next(iter(loaders["train"]))

    profiler.disable()
    stats = pstats.Stats(profiler).sort_stats("ncalls")
    stats.dump_stats("outputs/profiled_dataload")
    # NOTE: read it with snakeviz -s profiled_dataload (can be pip installed)

    # scene_dict = {
    #     "type": "scene",
    #     "integrator": rendering.define_integrator(),
    #     "camera": rendering.define_camera(3.5, 30, 60, "perspective", 512, 512),
    #     "emitter": rendering.define_emitter(None),
    #     "mesh": rendering.data_coloured_points_to_mitsuba(batch),
    # }
    # scene = mi.load_dict(scene_dict)

    # rendering.mega_kernel(False)
    # rendered_image = mi.render(scene)
    # rendering.flush_cache()

    # cols = batch.x[0]
    print("done")
